package org.productivity.java.syslog4j.impl.unix;

import org.productivity.java.syslog4j.SyslogRuntimeException;
import org.productivity.java.syslog4j.impl.AbstractSyslog;
import org.productivity.java.syslog4j.impl.AbstractSyslogWriter;
import org.productivity.java.syslog4j.util.OSDetectUtility;

import com.sun.jna.Library;
import com.sun.jna.Native;

/**
* UnixSyslog is an extension of AbstractSyslog that provides support for
* Unix-based syslog clients.
* 
* <p>This class requires the JNA (Java Native Access) library to directly
* access the native C libraries utilized on Unix platforms.</p>
* 
* <p>Syslog4j is licensed under the Lesser GNU Public License v2.1.  A copy
* of the LGPL license is available in the META-INF folder in all
* distributions of Syslog4j and in the base directory of the "doc" ZIP.</p>
* 
* @author &lt;syslog4j@productivity.org&gt;
* @version $Id: UnixSyslog.java,v 1.22 2009/06/04 03:33:49 cvs Exp $
*/
public class UnixSyslog extends AbstractSyslog {
	private static final long serialVersionUID = 4973353204252276740L;
	
	protected boolean libraryLoaded = false;
	
	protected UnixSyslogConfig unixSyslogConfig = null; 
	
    protected interface CLibrary extends Library {
        public void openlog(final String ident, int option, int facility);
        public void syslog(int priority, final String format, final String message);
        public void closelog();
    }
    
    protected CLibrary libraryInstance = null;
    protected boolean openlogCalled = false;

	protected synchronized void loadLibrary() throws SyslogRuntimeException {
		if (!OSDetectUtility.isUnix()) {
			throw new SyslogRuntimeException("UnixSyslog not supported on non-Unix platforms");
		}
		
		if (!this.libraryLoaded) {
			this.libraryInstance = (CLibrary) Native.loadLibrary(this.unixSyslogConfig.getLibrary(),CLibrary.class);
			this.libraryLoaded = true;
		}
	}

	public void initialize() throws SyslogRuntimeException {
		try {
			this.unixSyslogConfig = (UnixSyslogConfig) this.syslogConfig;
			
		} catch (ClassCastException cce) {
			throw new SyslogRuntimeException("config must be of type UnixSyslogConfig");
		}
		
		loadLibrary();	
	}
	
	protected void write(int level, String message) throws SyslogRuntimeException {
		int priority = this.unixSyslogConfig.getFacility() | level;

		synchronized(this) {
			if (!this.openlogCalled) {
				String ident = this.unixSyslogConfig.getIdent();
				
				if (ident != null && "".equals(ident.trim())) {
					ident = null;
				}
				
				this.libraryInstance.openlog(ident,this.unixSyslogConfig.getOption(),this.unixSyslogConfig.getFacility());
				this.openlogCalled = true;
			}
		}
		
		this.libraryInstance.syslog(priority,"%s",message);
	}
	
	protected void write(byte[] message) throws SyslogRuntimeException {
		// NO-OP
	}

	public void flush() throws SyslogRuntimeException {
		this.libraryInstance.closelog();
		this.openlogCalled = false;
	}

	public void shutdown() throws SyslogRuntimeException {
		flush();
	}

	public AbstractSyslogWriter getWriter() {
		return null;
	}

	public void returnWriter(AbstractSyslogWriter syslogWriter) {
		//
	}

	protected String prefixMessage(String message, String suffix) {
		 return message;
	}
}
